///////////////////////////////////////////////////////////////////////////////
// INCLUDES
///////////////////////////////////////////////////////////////////////////////

#include <stdint.h>
#include <stdbool.h>
#include <stdio.h>
#include <string.h>

#include "statemachine.h"
#include "util.h"

///////////////////////////////////////////////////////////////////////////////
// LOCAL MACROS
///////////////////////////////////////////////////////////////////////////////

#define STATES( FUN )      \
	FUN( Unauthenticated ) \
	FUN( Authenticated   ) \
	FUN( Locked          )

#define EFFECTS( FUN )    \
    FUN( PromptUsername ) \
    FUN( PromptPassword ) \
    FUN( DisplayWelcome ) \
    FUN( DisplayBye     ) \
    FUN( DisplayWarning ) \
    FUN( LogAttempt     ) \
    FUN( NotifyUser     )

#define GUARDS( FUN )   \
    FUN( NameExists  )  \
    FUN( PassCorrect )
    
#define TRIGGERS( FUN ) \
	FUN( Login    )     \
    FUN( Logout   )     \
    FUN( Lockdown )

#define STATEMACHINE                                                          \
	STATE( sUnauthenticated ),                                                \
        { tLogin      , NONE        , ePromptUsername   , CHOICE           }, \
        { EXTRA       , NONE        , ePromptPassword   , NONE             }, \
                                                                              \
        { CHOICE      , gNameExists , eDisplayWelcome   , sAuthenticated   }, \
        { EXTRA       , gPassCorrect, NONE              , NONE             }, \
                                                                              \
        { CHOICE      , gNameExists , eNotifyUser       , SAME             }, \
        { EXTRA       , NONE        , eLogAttempt       , NONE             }, \
                                                                              \
        { CHOICE      , ELSE        , eLogAttempt       , SAME             }, \
                                                                              \
    STATE_EX( sAuthenticated, NONE, eDisplayBye ),                            \
    	{ tLogout     , NONE        , NONE              , sUnauthenticated }, \
                                                                              \
    STATE_EX( sLocked, eDisplayWarning, NONE ),                               \
                                                                              \
    STATE( ANY ),                                                             \
    	{ tLockdown   , NONE        , NONE              , sLocked          } 	

#define INITIAL_STATE sUnauthenticated

#include "statemachine_functions.c"

///////////////////////////////////////////////////////////////////////////////
// LOCAL TYPES
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// LOCAL FUNCTION PROTOTYPES
///////////////////////////////////////////////////////////////////////////////

static void readline(char * buffer, size_t size);

///////////////////////////////////////////////////////////////////////////////
// LOCAL CONSTANTS AND VARIABLES
///////////////////////////////////////////////////////////////////////////////

#define BUFFERSIZE 16
char name[BUFFERSIZE];
char pass[BUFFERSIZE];

///////////////////////////////////////////////////////////////////////////////
// LOCAL FUNCTIONS
///////////////////////////////////////////////////////////////////////////////

// guards
static bool checkNameExists(void) {
	if (strcmp(name, "alice") == 0)
		return true;
	if (strcmp(name, "bob") == 0)
		return true;
		
	return false;
}

static bool checkPassCorrect(void) {
	if (strcmp(name, "alice") == 0) {
		if (strcmp(pass, "secret") == 0) {
			return true;
		}
	}
	if (strcmp(name, "bob") == 0) {
		if (strcmp(pass, "too short") == 0) {
			return true;
		}
	}
	
	return false;
}

// effects
static void handlePromptUsername(void) {
    printf("user name :");
	readline(name, BUFFERSIZE);
}

static void handlePromptPassword(void) {
    printf("password  :");
	readline(pass, BUFFERSIZE);
}

static void handleDisplayWelcome(void) {
    printf("Welcome %s!\n", name);
}

static void handleDisplayBye(void) {
    printf("Bye %s!\n", name);
}

static void handleDisplayWarning(void) {
    printf("WARNING: SYSTEM LOCKDOWN\n");
}

static void handleNotifyUser(void) {
    printf("User %s notified of login attempt.\n", name);
}

static void handleLogAttempt(void) {
    printf("Login attempt logged.\n");
}

static void handleAuthenticatedExit(void) {
    printf("Bye %s!\n", name);
}

static void readline(char * buffer, size_t size) {
	size_t i = 0;
	
	for(;;) {
		int c = getchar();
        if ( c < 0 )
            break;
        if ( c == '\n' )
            break;
        
        if (i < size) {
        	buffer[i++] = c;
        }
	}
	
	buffer[MIN(i, size - 1)] = '\0';
}

///////////////////////////////////////////////////////////////////////////////
// EXPORTED CONSTANTS AND VARIABLES
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// EXPORTED FUNCTIONS
///////////////////////////////////////////////////////////////////////////////

int main(void) {
    printf(
        "usage :\n"
        "   1) Login\n"
        "   0) Logout\n"
        "\n"
        "   z) Lockdown\n"
        "   x) exit\n");

    for(;;) {
        char input[3];
        readline(input, ARRAY_SIZE(input));
		if (input[1] != '\0')
        	continue; // more than 1 character input, ignore
        
        switch( input[0] ) {
            case '1' : applyTrigger(tLogin   ); break;
            case '0' : applyTrigger(tLogout  ); break;
            case 'z' : applyTrigger(tLockdown); break;
            case 'x' : return 0;
            default  : break;
        }
    }

    return 0;
}
