///////////////////////////////////////////////////////////////////////////////
// INCLUDES
///////////////////////////////////////////////////////////////////////////////

#include <stdint.h>
#include <stddef.h>
#include <assert.h>

#include "statemachine.h"

///////////////////////////////////////////////////////////////////////////////
// LOCAL MACROS
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// LOCAL TYPES
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// LOCAL FUNCTION PROTOTYPES
///////////////////////////////////////////////////////////////////////////////

static bool isStateRule(statemachineRule_t rule);
static bool evaluateGuards(statemachine_t *pStatemachine, size_t ruleIndex);
static void executeEffects(statemachine_t *pStatemachine, size_t ruleIndex);
static void executeEntryEffects(statemachine_t *pStatemachine, size_t ruleIndex);
static bool evaluateState(statemachine_t *pStatemachine, size_t ruleIndex);
static void evaluateChoices(statemachine_t *pStatemachine, size_t ruleIndex);
static void updateState(statemachine_t *pStatemachine, size_t ruleIndex);
static size_t findStateIndex(statemachine_t *pStatemachine, uint8_t state);

///////////////////////////////////////////////////////////////////////////////
// LOCAL CONSTANTS AND VARIABLES
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// LOCAL FUNCTIONS
///////////////////////////////////////////////////////////////////////////////

static bool isStateRule(statemachineRule_t rule) {
    return rule.trigger == INVALID;
}

static bool evaluateGuards(statemachine_t *pStatemachine, size_t ruleIndex) {
	assert( ruleIndex < pStatemachine->nrOfRules );
	
	guardHandler_t guardHandler = pStatemachine->guardHandler;	
	if (guardHandler == NULL) return true;
	
	size_t i = ruleIndex;
	do {
		statemachineRule_t rule = pStatemachine->pRules[i];
			
		if ( !guardHandler( rule.guard ) )
			return false;
		
		++i;
			
	} while (   (i < pStatemachine->nrOfRules)
	 	     && (pStatemachine->pRules[i].trigger == EXTRA));

	return true;
}

static void executeEffectsEx(statemachine_t *pStatemachine, size_t ruleIndex, bool guardAsEffect) {
	assert( ruleIndex < pStatemachine->nrOfRules );

    effectHandler_t effectHandler = pStatemachine->effectHandler;
    if (effectHandler == NULL) return;

	size_t i = ruleIndex;
	do {
		statemachineRule_t rule = pStatemachine->pRules[i];
		effectHandler( guardAsEffect ? rule.guard : rule.effect );
		++i;
			
	} while (   (i < pStatemachine->nrOfRules)
	 	     && (pStatemachine->pRules[i].trigger == EXTRA));

}

static void executeEffects(statemachine_t *pStatemachine, size_t ruleIndex) {
	executeEffectsEx(pStatemachine, ruleIndex, false);
}

static void executeEntryEffects(statemachine_t *pStatemachine, size_t ruleIndex) {
	executeEffectsEx(pStatemachine, ruleIndex, true);
}

static bool evaluateState(statemachine_t *pStatemachine, size_t stateIndex) {
	assert( stateIndex < pStatemachine->nrOfRules );
	assert( isStateRule(pStatemachine->pRules[stateIndex]) );

	uint8_t state        = pStatemachine->pRules[stateIndex].state;
	uint8_t currentState = pStatemachine->pRules[pStatemachine->currentStateIndex].state;
	
	return (state == currentState)
        || (state == ANY);
}

static void evaluateChoices(statemachine_t *pStatemachine, size_t ruleIndex) {
	assert( ruleIndex < pStatemachine->nrOfRules );

	for ( size_t i = ruleIndex + 1; i < pStatemachine->nrOfRules; ++i ) {
		statemachineRule_t rule = pStatemachine->pRules[i];
		if (rule.trigger == EXTRA)
			continue;
			
		if (rule.trigger == CHOICE) {
			if ( evaluateGuards(pStatemachine, i) ) {
				executeEffects(pStatemachine, i);
				
				updateState(pStatemachine, i);
				return;
			}
		}
	}
	
	assert( false );
}

static size_t findStateIndex(statemachine_t *pStatemachine, uint8_t state) {
   for ( size_t i = 0; i < pStatemachine->nrOfRules; ++i ) {
		statemachineRule_t rule = pStatemachine->pRules[i];
        if ( isStateRule(rule) && (rule.state == state) ) {
			return i;
        }
	}
	
	assert( false );
	return INVALID;
}

static void updateState(statemachine_t *pStatemachine, size_t ruleIndex) {
	assert( ruleIndex < pStatemachine->nrOfRules );
	
    statemachineRule_t rule = pStatemachine->pRules[ruleIndex];
	uint8_t currentState = pStatemachine->pRules[pStatemachine->currentStateIndex].state;
	
    if ( (rule.state == SAME) || (rule.state == currentState) )
    	return;
    
    // execute exit functions
    executeEffects(pStatemachine, pStatemachine->currentStateIndex);

	// execute entry functions and update index
	size_t newIndex = findStateIndex(pStatemachine, rule.state);
	executeEntryEffects(pStatemachine, newIndex);
	pStatemachine->currentStateIndex = newIndex;
}

///////////////////////////////////////////////////////////////////////////////
// EXPORTED CONSTANTS AND VARIABLES
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// EXPORTED FUNCTIONS
///////////////////////////////////////////////////////////////////////////////

void statemachine_ApplyTrigger(
    statemachine_t *pStatemachine,
    uint8_t         trigger      ) {

    size_t internalStateIndex = INVALID;

    for ( size_t i = pStatemachine->currentStateIndex; i < pStatemachine->nrOfRules; ++i ) {
        statemachineRule_t rule = pStatemachine->pRules[i];
        
        if ( isStateRule(rule) ) {
            internalStateIndex = i;
            continue;
        }
        
        if ( rule.trigger == EXTRA )
        	continue;
        	
        if ( rule.trigger == CHOICE )
        	continue;
        
        if (   evaluateState(pStatemachine, internalStateIndex)
            && (trigger == rule.trigger)
            && (evaluateGuards(pStatemachine, i))) {

			executeEffects(pStatemachine, i);
            
            if (rule.state == CHOICE) {
            	evaluateChoices(pStatemachine, i);
            } else {
				updateState(pStatemachine, i);
            }
                        
            break;
        }
    }
}
