// Example statemachine implementation and usage.
//

///////////////////////////////////////////////////////////////////////////////
// INCLUDES
///////////////////////////////////////////////////////////////////////////////

#include <stdint.h>
#include <stdbool.h>

///////////////////////////////////////////////////////////////////////////////
// MACROS
///////////////////////////////////////////////////////////////////////////////

#define INVALID 	        0xFFu
#define EXTRA               0xFEu
#define CHOICE              0xFDu

#define STATE_ENTRY_EXIT    0xFCu

#define STATE( s ) 	        { INVALID, INVALID, INVALID, s }
#define STATE_EX( s, a, b ) { INVALID, a      , b      , s }

#define NONE 		INVALID
#define ELSE 		INVALID
#define SAME 		INVALID
#define ANY 		INVALID

#define TRIGGER_ENUM_ITEM( item ) t##item,
#define GUARD_ENUM_ITEM( item ) g##item,
#define EFFECT_ENUM_ITEM( item ) e##item,
#define STATE_ENUM_ITEM( item ) s##item,

// create a function prototype from the effect name with prefix 'handle'
#define EFFECT_PROTOTYPE_ITEM( item ) static void handle##item(void);

// create a case statement for effect in which the corresponding function is
// called
#define EFFECT_CASE_ITEM( item ) case e##item : handle##item(); break;

// add 'g' prefix to the guard name
#define GUARD_ENUM_ITEM( item ) g##item,

// create a function prototype from the guard name with prefix 'handle'
#define GUARD_PROTOTYPE_ITEM( item ) static bool check##item(void);

// create a case statement for effect in which the corresponding guard function
// is called
#define GUARD_CASE_ITEM( item ) case g##item : return check##item();


///////////////////////////////////////////////////////////////////////////////
// TYPES
///////////////////////////////////////////////////////////////////////////////

typedef bool (*guardHandler_t)(uint8_t guard);
typedef void (*effectHandler_t)(uint8_t effect);

typedef struct statemachineRule_t {
    uint8_t trigger;
    uint8_t guard;
    uint8_t effect;
    uint8_t state;
} statemachineRule_t;

typedef struct statemachine_t {
    guardHandler_t              guardHandler;
    effectHandler_t             effectHandler;
    const statemachineRule_t    *pRules;
    const uint8_t               nrOfRules;
    uint8_t                     currentStateIndex;
} statemachine_t;

///////////////////////////////////////////////////////////////////////////////
// FUNCTION PROTOTYPES
///////////////////////////////////////////////////////////////////////////////

void statemachine_ApplyTrigger(
    statemachine_t  *pstatemachine  ,
    const uint8_t   trigger         );

///////////////////////////////////////////////////////////////////////////////
// CONSTANTS AND VARIABLES
///////////////////////////////////////////////////////////////////////////////
